package lsfusion.server.physics.admin.backup.action;

import com.google.common.base.Throwables;
import lsfusion.base.col.MapFact;
import lsfusion.base.col.interfaces.immutable.ImMap;
import lsfusion.base.col.interfaces.immutable.ImOrderMap;
import lsfusion.base.col.interfaces.immutable.ImRevMap;
import lsfusion.base.col.interfaces.immutable.ImSet;
import lsfusion.base.file.IOUtils;
import lsfusion.interop.session.ExternalUtils;
import lsfusion.server.data.expr.key.KeyExpr;
import lsfusion.server.data.query.build.QueryBuilder;
import lsfusion.server.data.sql.exception.SQLHandledException;
import lsfusion.server.data.value.DataObject;
import lsfusion.server.data.value.ObjectValue;
import lsfusion.server.language.ScriptingErrorLog;
import lsfusion.server.language.ScriptingLogicsModule;
import lsfusion.server.logics.action.Action;
import lsfusion.server.logics.action.controller.context.ExecutionContext;
import lsfusion.server.logics.classes.user.ConcreteCustomClass;
import lsfusion.server.logics.property.Property;
import lsfusion.server.logics.property.classes.ClassPropertyInterface;
import lsfusion.server.physics.dev.integration.internal.to.InternalAction;
import lsfusion.server.physics.exec.db.controller.manager.DBManager;
import org.apache.commons.io.FilenameUtils;

import java.sql.SQLException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;

import static lsfusion.base.BaseUtils.splitTrim;

public class BackupAction extends InternalAction {

    public BackupAction(ScriptingLogicsModule LM) {
        super(LM);
    }

    public void executeInternal(ExecutionContext<ClassPropertyInterface> context) {
        makeBackup(context, false);
    }

    protected void makeBackup(ExecutionContext context, boolean partial) {
        DBManager dbManager = context.getDbManager();
        if (dbManager.checkBackupParams(context)) {

            try (ExecutionContext.NewSession newContext = context.newSession()) {

                Integer threadCount = (Integer) findProperty("threadCount[]").read(newContext);
                if (threadCount == null || threadCount < 1) {
                    threadCount = 1;
                }

                String backupFileName = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd-HH-mm-ss"));

                List<String> excludeTables = partial ? getExcludeTables(context) : new ArrayList<>();
                String extraExcludeTablesString = partial ? getExtraExcludeTables(context) : null;
                List<String> extraExcludeTables = splitTrim(extraExcludeTablesString);

                String backupFilePath = dbManager.getBackupFilePath(backupFileName);
                String backupFileLogPath = dbManager.getBackupFileLogPath(backupFileName);

                DataObject backupObject = newContext.addObject((ConcreteCustomClass) findClass("Backup"));
                LocalDateTime currentDateTime = LocalDateTime.now();
                findProperty("date[Backup]").change(currentDateTime.toLocalDate(), newContext, backupObject);
                findProperty("time[Backup]").change(currentDateTime.toLocalTime(), newContext, backupObject);
                findProperty("file[Backup]").change(backupFilePath, newContext, backupObject);
                findProperty("name[Backup]").change(FilenameUtils.getName(backupFilePath), newContext, backupObject);
                findProperty("fileLog[Backup]").change(backupFileLogPath, newContext, backupObject);
                findProperty("isMultithread[Backup]").change(threadCount > 1, newContext, backupObject);

                if (partial) {
                    findProperty("partial[Backup]").change(true, newContext, backupObject);
                    for (String excludeTable : excludeTables) {
                        ObjectValue tableObject = findProperty("table[ISTRING[100]]").readClasses(newContext, new DataObject(excludeTable));
                        if (tableObject instanceof DataObject)
                            findProperty("exclude[Backup,Table]").change(true, newContext, backupObject, (DataObject) tableObject);
                    }
                    findProperty("extraExclude[Backup]").change(extraExcludeTablesString, newContext, backupObject);
                }

                newContext.apply();

                backupObject = new DataObject((Long) backupObject.object, (ConcreteCustomClass) findClass("Backup")); // обновляем класс после backup

                excludeTables.addAll(extraExcludeTables);
                dbManager.backupDB(context, backupFileName, threadCount, excludeTables);

                findProperty("log[Backup]").change(IOUtils.readFileToString(backupFileLogPath, ExternalUtils.resourceCharset.name()), newContext, backupObject);
                newContext.apply();
            } catch (Exception e) {
                throw Throwables.propagate(e);
            }
        }
    }

    private List<String> getExcludeTables(ExecutionContext context) throws ScriptingErrorLog.SemanticErrorException, SQLException, SQLHandledException {
        KeyExpr tableExpr = new KeyExpr("Table");
        ImRevMap<Object, KeyExpr> tableKeys = MapFact.<Object, KeyExpr>singletonRev("Table", tableExpr);

        QueryBuilder<Object, Object> tableQuery = new QueryBuilder<>(tableKeys);
        tableQuery.addProperty("sidTable", findProperty("sid[Table]").getExpr(context.getModifier(), tableExpr));
        tableQuery.and(findProperty("exclude[Table]").getExpr(context.getModifier(), tableExpr).getWhere());

        ImOrderMap<ImMap<Object, Object>, ImMap<Object, Object>> tableResult = tableQuery.execute(context.getSession());

        List<String> excludeTables = new ArrayList<>();
        for (ImMap<Object, Object> entry : tableResult.values()) {

            String sidTable = (String) entry.get("sidTable");
            if (sidTable != null)
                excludeTables.add(sidTable.trim());
        }
        return excludeTables;
    }

    private String getExtraExcludeTables(ExecutionContext context) throws ScriptingErrorLog.SemanticErrorException, SQLException, SQLHandledException {
        return (String) findProperty("extraExclude[]").read(context);
    }

    @Override
    public ImMap<Property, Boolean> aspectChangeExtProps(ImSet<Action<?>> recursiveAbstracts) {
        try {
            return getChangeProps(findProperty("date[Backup]").property, findProperty("time[Backup]").property);
        } catch (ScriptingErrorLog.SemanticErrorException e) {
            return null;
        }
    }
}
